<?php
/* $Id: csstats.class.php 12 2010-05-18 12:13:01Z kajoj $ */

/*
 * (C) Copyright 2009 Karol 'kajoj' Nowacki <kajoj@xt24.eu>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * See LICENSE.TXT file for more information.
 *
 */

/*
 * DESCRIPTION :
 * This is PHP class for handle files csstats.dat generated by CSX module
 * in AMX Mod X (http://www.amxmodx.org/).
 *
 */

/**
 * This is PHP class for handle files csstats.dat generated by CSX module
 * in AMX Mod X (http://www.amxmodx.org/).<br /><br />
 * <b>supported (tested) file version</b>
 * <ul>
 * <li><b>0x0b</b> - AMXmodX version 1.8.1</li>
 * </ul><br />
 *
 * <b>CSstats main features are:</b>
 * <ul>
 * <li>read information form csstats.dat files</li>
 * </ul><br />
 *
 * <b>TODO:</b>
 * <ul>
 * </ul>
 * @package CSstats
 * @abstract Class for handle csstats.dat file generated by CSX module in AMX Mod X.
 * @author kajoj
 * @copyright 2009 Karol 'kajoj' Nowacki <kajoj@xt24.eu>
 * @license http://www.gnu.org/copyleft/lesser.html LGPL
 * @version 0.4
 */

/**
 * CHANGELOG:
 * 0.4
 *    + search players
 *    * fix PDO export update (duplicate rows)
 * 0.3:
 * 	  + array access for read implementation
 *    + interator implementation
 *    * fix all file read bug
 */

if (!class_exists('CSstats', false)) {


	/**
	 * This is main class for handle csstats.dat files
	 * @author kajoj
	 * @version 0.4
	 * @package CSstats
	 */
	class CSstats implements Iterator, ArrayAccess {
		
		/**
		 * List of player info fields
		 * @var array
		 */
		public static $player_info_fields = array('rank', 'nick', 'uniq', 'teamkill', 'damage', 'deaths', 'kills', 'shots', 'hits', 'headshots',
			'defusions', 'defused', 'plants', 'explosions',
			'head', 'chest', 'stomach', 'leftarm', 'rightarm', 'leftleg', 'rightleg'); 

		// protected or Protected properties

		/**
		 * @var array array of players
		 * @access protected
		 */
		protected $players = array();

		/**
		 * @var int version of file csstats.dat
		 * @access protected
		 */
		protected $fileversion = 0x00;

		/**
		 * @var string path to imported file
		 * @access protected
		 */
		protected $path = "";

		/**
		 * @var int current interator position
		 * @access protected
		 * @since 0.3
		 */
		protected $position = 0;

		/**
		 * This is class constructor.
		 * @param string $path path to the file to be loaded. By default empty string - nothig to read
		 * @param int $maxplayers limit the number of players to load. By default 0 - unlimited
		 */
		public function __construct($path = "", $maxplayers = 0) {

			$this->path = $path;
			$this->fileversion = 0x00;
			$this->players = array();
			$this->position = 1;
				
			if (empty($path))
				return;

			if (!file_exists($path))
				throw new CSstatsException("File not found.");
			if (!is_readable($path))
				throw new CSstatsException("File is not readable.");
			$handle = @fopen($path, "r");
			if ($handle === false)
				throw new CSstatsException("Can not open file.");

			try {
				$this->fileversion = $this->readShortData($handle);

				$i = 1; $player = array();
				while (!is_null($player) && !feof($handle) && ($maxplayers == 0 || $i<=$maxplayers)) {
					$player = $this->readPlayer($handle);
					if (!is_null($player)) {
						$player['rank'] = $i;	
						$this->players[$i] = $player;
					}
					$i++;
				}

			} catch (CSstatsException $e) {
				fclose($handle);
				throw $e;
			}

			fclose($handle);
		}

		/**
		 * This method read player information form file.
		 * @param $handle open file handle from which to read the player.
		 * @return array of player information
		 * @access protected
		 */
		protected function readPlayer($handle) {
			$p = array();
			$l = $this->readShortData($handle);
			if ($l == 0)
				return null;
			$p['nick'] = $this->readStringData($handle, $l);
			$l = $this->readShortData($handle);
			$p['uniq'] = $this->readStringData($handle, $l);

			$p['teamkill'] = $this->readIntData($handle);
			$p['damage'] = $this->readIntData($handle);
			$p['deaths'] = $this->readIntData($handle);
			$p['kills'] = $this->readIntData($handle);
			$p['shots'] = $this->readIntData($handle);
			$p['hits'] = $this->readIntData($handle);
			$p['headshots'] = $this->readIntData($handle);

			$p['defusions'] = $this->readIntData($handle);
			$p['defused'] = $this->readIntData($handle);
			$p['plants'] = $this->readIntData($handle);
			$p['explosions'] = $this->readIntData($handle);

			$this->readIntData($handle); // 0x00000000

			$p['head'] = $this->readIntData($handle);
			$p['chest'] = $this->readIntData($handle);
			$p['stomach'] = $this->readIntData($handle);
			$p['leftarm'] = $this->readIntData($handle);
			$p['rightarm'] = $this->readIntData($handle);
			$p['leftleg'] = $this->readIntData($handle);
			$p['rightleg'] = $this->readIntData($handle);

			$this->readIntData($handle); // 0x00000000

			return $p;
		}

		/**
		 * This method read integer (32bit) data from file.
		 * @param $handle open file handle from which to read data
		 * @return int
		 * @access protected
		 */
		protected function readIntData($handle) {
			$d = @fread($handle, 4);
			if ($d === false)
				throw new CSstatsException("Can not read data.");
			$d = unpack("V", $d);
			return $d[1];
		}

		/**
		 * This method read short integer (16bit) data form file.
		 * @param $handle open file handle from which to read data
		 * @return int
		 * @access protected
		 */
		protected function readShortData($handle) {
			$d = @fread($handle, 2);
			if ($d === false)
				throw new CSstatsException("Can not read data.");
			$d = unpack("v", $d);
			return $d[1];
		}

		/**
		 * This method read string data from file.
		 * @param $handle open file handle from which to read data
		 * @param $len length of string to read (with ending NULL byte)
		 * @return string
		 * @access protected
		 */
		protected function readStringData($handle, $len) {
			$d = @fread($handle, $len);
			if ($d === false)
				throw new CSstatsException("Can not read data.");
			$d = trim($d);
			return $d;
		}

		/**
		 * This method return player information placed on $id rank
		 * @param $id Player ranking position
		 * @return array
		 */
		public function getPlayer($id) {
			if (isset($this->players[$id]))
			return $this->players[$id];
			return null;
		}
		
		/**
		 * This method return number of loaded players
		 * @return int
		 */
		public function countPlayers() {
			return count($this->players);
		}
		
		
		/**
		 * This method is useful for export csstats data to otger data collectors.
		 * It allow save data in Database via PDO object and some basic text files. 
		 * @param $target mixed Where to save data (PDO Object or path to file)
		 * @param $options array Array of exporting options
		 * @return bool true if success
		 */
		public function export($target, $options = array()) {					
			if (class_exists('PDO', false)) {
				if ($target instanceof PDO)
					return $this->exportViaPDO($target, $options);			
			}
			if (is_string($target)) {
				return $this->exportTofile($target, $options);
			} 
			return false;
		}
		
		protected function exportViaPDO($target, $options = array()) {
			if (!isset($options['table']))
				return false;
			if (!isset($options['fields']))
				return false;
			if (!is_array($options['fields']))
				return false;
			if (count($options['fields']) == 0)
				return false;
			foreach (array_keys($options['fields']) as $k)
				if (!in_array($k, self::$player_info_fields))
					return false;
			
			$target->beginTransaction();
			$INSstmt = $target->prepare('INSERT INTO `'.$options['table'].'` '.
				' ( '.join(", ", $options['fields']).' ) VALUES ( :'.join(", :", array_keys($options['fields'])).' )');

			if (isset($options['key']) && in_array($options['key'], array_keys($options['fields']))) {
				$f = array();
				foreach ($options['fields'] as $k => $v)
					$f[]= "{$v} = :{$k}";
				
				$UPDstmt = $target->prepare("UPDATE `{$options['table']}` SET ".join(", ", $f)." WHERE {$options['fields'][$options['key']]} = :{$options['key']}");
				$SELstmt = $target->prepare("SELECT COUNT(*) FROM `{$options['table']}` WHERE {$options['fields'][$options['key']]} = :{$options['key']}");	
			}
			
			foreach ($this->players as $p) {
				if (isset($UPDstmt)) {
					foreach(array_keys($options['fields']) as $k)
						$UPDstmt->bindValue(':'.$k, $p[$k]);
					$UPDstmt->execute();
					if ($UPDstmt->rowCount() > 0)
						continue;
	
					$SELstmt->bindValue(':'.$options['key'], $p[$options['key']]);
					$SELstmt->execute();
					if (current($SELstmt->fetch(PDO::FETCH_NUM)) > 0)
						continue;	
				}
				foreach(array_keys($options['fields']) as $k)
					$INSstmt->bindValue(':'.$k, $p[$k]);
				if (!$INSstmt->execute()) {
					$target->rollBack();
					return false;
				}
			}
			$target->commit();
			return true;
		}
		
		protected function exportTofile($target, $options = array()) {
			if (strlen($target) == 0)
				return false;
			
			$filetype = "txt";
			if (isset($options['filetype']))
				$filetype = $options['filetype'];
			if (!in_array($filetype, array('txt', 'csv')))
				return false;
				
			if (!isset($options['fields']))
				return false;
			if (!is_array($options['fields']))
				return false;
			if (count($options['fields']) == 0)
				return false;
			foreach ($options['fields'] as $k)
				if (!in_array($k, self::$player_info_fields))
					return false;

			$enclosure = '"';
			if (isset($option['enclosure']))
				$enclosure = $option['enclosure'];
					
			$content = "";
			foreach ($this->players as $player) {
				$v = array();
				foreach($options['fields'] as $k)
					$v[] = $player[$k];
					
				switch ($filetype) {
					case "txt" :
						$content .= join(isset($options['delimiter'])?$options['delimiter']:"\t", $v);
						break;
					case "csv":
						
						foreach ($v as $k => $vv)
							if (!is_int($vv))
								$v[$k] = $enclosure.str_replace(array("\\", $enclosure), array("\\\\", "\\".$enclosure), $vv).$enclosure;
								
						$content .= join(isset($options['delimiter'])?$options['delimiter']:',', $v); 
						break;
				}
				$content .= "\n";
			}
			return @file_put_contents($target, $content);
		}
		
		/**
		 * This method searches the list of players by Nick using a regular expression,
		 * or strict comparison. Returns an array of matching players.
		 * @param $pattern string Regular expresion or searching nickname
		 * @param $regexp boolean use regular exxpression?
		 * @return array Array of matching players
		 * @since 0.4
		 */
		public function searchByNick($pattern, $useRegexp = false) {
			$find = array();
			foreach ($this->players as $player)
				if ($useRegexp && preg_match($pattern, $player['nick']) || (!$useRegexp && $pattern == $player['nick']))
					$find[] = $player;
			return $find;
		}

		//Interator implementation

		public function rewind() {
			$this->position = 1;
		}

		public function current() {
			return $this->players[$this->position];
		}

		public function key() {
			return $this->position;
		}

		public function next() {
			++$this->position;
		}

		public function valid() {
			return isset($this->players[$this->position]);
		}

		//End of Interator implementation
		
		//ArrayAccess implementation
		public function offsetSet($offset, $value) {
			//TODO: implement set player information method
		}

		public function offsetExists($offset) {
			return isset($this->players[$offset]);
		}
		
		public function offsetUnset($offset) {
			//TODO: implement delete player method
		}

		public function offsetGet($offset) {
			return isset($this->players[$offset]) ? $this->players[$offset] : null;
		}
		//End of ArratAccess implementation
	}
}

if (!class_exists('CSstatsException', false)) {
	/**
	 * This is exception class for CSstats class.
	 * @author kajoj
	 * @package CSstats
	 */
	class CSstatsException extends Exception {}
}
